#! /usr/bin/python3
import os
import sys
import generator_utils as gen

template = """// @{generatedby}@
/* ///////////////////////// The MPI Bugs Initiative ////////////////////////

  Origin: MBI

  Description: @{shortdesc}@
    @{longdesc}@

   Version of MPI: Conforms to MPI 1.1, does not require MPI 2 implementation

BEGIN_MPI_FEATURES
  P2P!basic: @{p2pfeature}@
  P2P!nonblocking: @{ip2pfeature}@
  P2P!persistent: @{persfeature}@
  COLL!basic: Lacking
  COLL!nonblocking: Lacking
  COLL!persistent: Lacking
  COLL!tools: Lacking
  RMA: Lacking
END_MPI_FEATURES

BEGIN_MBI_TESTS
  $ mpirun -np 2 ${EXE}
  | @{outcome}@
  | @{errormsg}@
END_MBI_TESTS
//////////////////////       End of MBI headers        /////////////////// */

#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>


int main(int argc, char **argv) {
  int nprocs = -1;
  int rank = -1;
  int src=0, dest=1;
  int stag=0, rtag=0;
  int buff_size = 1;

  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD, &nprocs);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  printf("Hello from rank %d \\n", rank);

  if (nprocs < 2)
    printf("MBI ERROR: This test needs at least 2 processes to produce a bug!\\n");

  MPI_Comm newcom = MPI_COMM_WORLD;
  MPI_Datatype type = MPI_INT;

  @{change_arg}@

  @{init1}@
  @{init2}@
  if (rank == 0) {
    @{operation1}@ /* MBIERROR1 */
    @{start1}@
    @{fini1}@
  }else if (rank == 1) {
    @{operation2}@ /* MBIERROR2 */
    @{start2}@
    @{fini2}@
  }
  @{free1}@
  @{free2}@

  MPI_Finalize();
  printf("Rank %d finished normally\\n", rank);
  return 0;
}
"""

##################################
# Generate code with type mismatch
##################################

for p1 in gen.allsend:
    for p2 in gen.allrecv:
        patterns = {}
        patterns = {'p1': p1, 'p2': p2}
        patterns['generatedby'] = f'DO NOT EDIT: this file was generated by {os.path.basename(sys.argv[0])}. DO NOT EDIT.'
        patterns['p2pfeature'] = 'Yes' if p1 in gen.send + gen.ssend + gen.bsend or p2 in gen.recv  else 'Lacking'
        patterns['ip2pfeature'] = 'Yes' if p1 in gen.isend or p2 in gen.irecv  else 'Lacking'
        patterns['persfeature'] = 'Yes' if p1 in gen.psend or p2 in gen.precv  else 'Lacking'
        patterns['p1'] = p1
        patterns['p2'] = p2
        patterns['init1'] = gen.init[p1]("1")
        patterns['init2'] = gen.init[p2]("2")
        patterns['start1'] = gen.start[p1]("1")
        patterns['start2'] = gen.start[p2]("2")
        patterns['fini1'] = gen.fini[p1]("1")
        patterns['fini2'] = gen.fini[p2]("2")
        patterns['operation1'] = gen.operation[p1]("1") #send
        patterns['operation2'] = gen.operation[p2]("2") #recv
        patterns['free1'] = gen.free[p1]("1")
        patterns['free2'] = gen.free[p2]("2")

        # Generate the incorrect matching
        replace = patterns.copy()
        replace['shortdesc'] = 'Point to point @{p1}@ and @{p2}@ have a datatype mismatch'
        replace['longdesc'] = 'Process 0 uses MPI_FLOAT as the datatype while process 1 uses MPI_INT.'
        replace['outcome'] = 'ERROR: DatatypeMatching'
        replace['errormsg'] = 'P2P Datatype mismatch. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ and @{p2}@ at @{filename}@:@{line:MBIERROR2}@ have MPI_INT and MPI_FLOAT as a datatype'
        replace['change_arg'] = 'if (rank == 0)\n    type = MPI_FLOAT; /* MBIERROR3 */'
        gen.make_file(template, f'ParamMatching_Data_{p1}_{p2}_nok.c', replace)

        # Generate code with a null type
        replace = patterns.copy()
        replace['shortdesc'] = 'Use of invalid datatype in point-to-point communication'
        replace['longdesc'] = 'Point to point @{p1}@ and @{p2}@ have MPI_DATATYPE_NULL as a type'
        replace['outcome'] = 'ERROR: InvalidDatatype'
        replace['errormsg'] = 'Invalid datatype in P2P. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ and @{p2}@ at @{filename}@:@{line:MBIERROR2}@ have MPI_DATATYPE_NULL as a type'
        replace['change_arg'] = 'type = MPI_DATATYPE_NULL; /* MBIERROR3 */'
        gen.make_file(template, f'InvalidParam_DatatypeNull_{p1}_{p2}_nok.c', replace)

        # Generate code with an invalid datatype
        replace = patterns.copy()
        replace['shortdesc'] = 'Use of invalid datatype in point-to-point communication'
        replace['longdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid datatype'
        replace['outcome'] = 'ERROR: InvalidDatatype'
        replace['errormsg'] = 'Invalid datatype in P2P. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ and @{p2}@ at @{filename}@:@{line:MBIERROR2}@ have an invalid datatype'
        replace['change_arg'] = 'MPI_Type_contiguous (2, MPI_INT, &type); MPI_Type_commit(&type);MPI_Type_free(&type); /* MBIERROR3 */'
        gen.make_file(template, f'InvalidParam_Datatype_{p1}_{p2}_nok.c', replace)

#################################
# Generate code with tag mismatch
#################################

for p1 in gen.allsend:
    for p2 in gen.allrecv:
        patterns = {}
        patterns = {'p1': p1, 'p2': p2}
        patterns['generatedby'] = f'DO NOT EDIT: this file was generated by {os.path.basename(sys.argv[0])}. DO NOT EDIT.'
        patterns['p2pfeature'] = 'Yes' if p1 in gen.send + gen.ssend + gen.bsend or p2 in gen.recv  else 'Lacking'
        patterns['ip2pfeature'] = 'Yes' if p1 in gen.isend or p2 in gen.irecv  else 'Lacking'
        patterns['persfeature'] = 'Yes' if p1 in gen.psend or p2 in gen.precv  else 'Lacking'
        patterns['p1'] = p1
        patterns['p2'] = p2
        patterns['init1'] = gen.init[p1]("1")
        patterns['init2'] = gen.init[p2]("2")
        patterns['start1'] = gen.start[p1]("1")
        patterns['start2'] = gen.start[p2]("2")
        patterns['fini1'] = gen.fini[p1]("1")
        patterns['fini2'] = gen.fini[p2]("2")
        patterns['operation1'] = gen.operation[p1]("1") #send
        patterns['operation2'] = gen.operation[p2]("2") #recv
        patterns['free1'] = gen.free[p1]("1")
        patterns['free2'] = gen.free[p2]("2")
        patterns['change_arg'] = ""

        # Generate the incorrect tag matching
        replace = patterns.copy()
        replace['shortdesc'] = 'Point to point @{p1}@ and @{p2}@ have a tag mismatch'
        replace['longdesc'] = 'Point to point @{p1}@ and @{p2}@ have a tag mismatch.'
        replace['outcome'] = 'ERROR: TagMatching'
        replace['errormsg'] = 'P2P tag mismatch. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ and @{p2}@ at @{filename}@:@{line:MBIERROR2}@ use different tag.'
        replace['change_arg'] = 'stag=0; rtag=1;/* MBIERROR */'
        gen.make_file(template, f'ParamMatching_Tag_{p1}_{p2}_nok.c', replace)

        # Generate the code with an invalid tag
        replace = patterns.copy()
        replace['shortdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid tag'
        replace['longdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid tag.'
        replace['outcome'] = 'ERROR: InvalidTag'
        replace['errormsg'] = 'Invalid Tag. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ and @{p2}@ at @{filename}@:@{line:MBIERROR2}@ use an invalid tag.'
        replace['change_arg'] = 'stag=-1; rtag=-2;/* MBIERROR */'
        gen.make_file(template, f'InvalidParam_Tag_{p1}_{p2}_nok.c', replace)

        # Generate a correct code using MPI_ANY_TAG
        replace = patterns.copy()
        replace['shortdesc'] = 'Correct code'
        replace['longdesc'] = 'Correct code'
        replace['outcome'] = 'OK'
        replace['errormsg'] = 'OK'
        replace['change_arg'] = 'rtag=MPI_ANY_TAG;'
        gen.make_file(template, f'ParamMatching_Tag_{p1}_{p2}_ok.c', replace)
