from pypy.module.micronumpy.test.test_base import BaseNumpyAppTest

class AppTestSorting(BaseNumpyAppTest):
    def test_argsort_dtypes(self):
        from numpypy import array, arange
        assert array(2.0).argsort() == 0
        nnp = self.non_native_prefix
        for dtype in ['int', 'float', 'int16', 'float32', 'uint64',
                      nnp + 'i2', complex]:
            a = array([6, 4, -1, 3, 8, 3, 256+20, 100, 101], dtype=dtype)
            exp = list(a)
            exp = sorted(range(len(exp)), key=exp.__getitem__)
            c = a.copy()
            res = a.argsort()
            assert (res == exp).all(), '%r\n%r\n%r' % (a,res,exp)
            assert (a == c).all() # not modified

            a = arange(100, dtype=dtype)
            assert (a.argsort() == a).all()
        import sys
        if '__pypy__' in sys.builtin_module_names:
            raises(NotImplementedError, 'arange(10,dtype="float16").argsort()')

    def test_argsort_ndim(self):
        from numpypy import array
        a = array([[4, 2], [1, 3]])
        assert (a.argsort() == [[1, 0], [0, 1]]).all()
        a = array(range(10) + range(10) + range(10))
        b = a.argsort()
        assert (b[:3] == [0, 10, 20]).all()
        #trigger timsort 'run' mode which calls arg_getitem_slice
        a = array(range(100) + range(100) + range(100))
        b = a.argsort()
        assert (b[:3] == [0, 100, 200]).all()
        a = array([[[]]]).reshape(3,4,0)
        b = a.argsort()
        assert b.size == 0

    def test_argsort_random(self):
        from numpypy import array
        from _random import Random
        rnd = Random(1)
        a = array([rnd.random() for i in range(512*2)]).reshape(512,2)
        a.argsort()

    def test_argsort_axis(self):
        from numpypy import array
        a = array([[4, 2], [1, 3]])
        assert (a.argsort(axis=None) == [2, 1, 3, 0]).all()
        assert (a.argsort(axis=-1) == [[1, 0], [0, 1]]).all()
        assert (a.argsort(axis=0) == [[1, 0], [0, 1]]).all()
        assert (a.argsort(axis=1) == [[1, 0], [0, 1]]).all()
        a = array([[3, 2, 1], [1, 2, 3]])
        assert (a.argsort(axis=0) == [[1, 0, 0], [0, 1, 1]]).all()
        assert (a.argsort(axis=1) == [[2, 1, 0], [0, 1, 2]]).all()

    def test_sort_dtypes(self):
        from numpypy import array, arange
        for dtype in ['int', 'float', 'int16', 'float32', 'uint64',
                      'i2', complex]:
            a = array([6, 4, -1, 3, 8, 3, 256+20, 100, 101], dtype=dtype)
            exp = sorted(list(a))
            res = a.copy()
            res.sort()
            assert (res == exp).all(), '%r\n%r\n%r' % (a,res,exp)

            a = arange(100, dtype=dtype)
            c = a.copy()
            a.sort()
            assert (a == c).all()

    def test_sort_nonnative(self):
        from numpypy import array
        nnp = self.non_native_prefix
        for dtype in [ nnp + 'i2']:
            a = array([6, 4, -1, 3, 8, 3, 256+20, 100, 101], dtype=dtype)
            b = array([-1, 3, 3, 4, 6, 8, 100, 101, 256+20], dtype=dtype)
            c = a.copy()
            import sys
            if '__pypy__' in sys.builtin_module_names:
                exc = raises(NotImplementedError, a.sort)
                assert exc.value[0].find('supported') >= 0
            #assert (a == b).all(), \
            #    'a,orig,dtype %r,%r,%r' % (a,c,dtype)

# tests from numpy/tests/test_multiarray.py
    def test_sort_corner_cases(self):
        # test ordering for floats and complex containing nans. It is only
        # necessary to check the lessthan comparison, so sorts that
        # only follow the insertion sort path are sufficient. We only
        # test doubles and complex doubles as the logic is the same.

        # check doubles
        from numpypy import array, zeros, arange
        from math import isnan
        nan = float('nan')
        a = array([nan, 1, 0])
        b = a.copy()
        b.sort()
        assert [isnan(bb) for bb in b] == [isnan(aa) for aa in a[::-1]]
        assert (b[:2] == a[::-1][:2]).all()

        b = a.argsort()
        assert (b == [2, 1, 0]).all()

        # check complex
        a = zeros(9, dtype='complex128')
        a.real += [nan, nan, nan, 1, 0, 1, 1, 0, 0]
        a.imag += [nan, 1, 0, nan, nan, 1, 0, 1, 0]
        b = a.copy()
        b.sort()
        assert [isnan(bb) for bb in b] == [isnan(aa) for aa in a[::-1]]
        assert (b[:4] == a[::-1][:4]).all()

        b = a.argsort()
        assert (b == [8, 7, 6, 5, 4, 3, 2, 1, 0]).all()

        # all c scalar sorts use the same code with different types
        # so it suffices to run a quick check with one type. The number
        # of sorted items must be greater than ~50 to check the actual
        # algorithm because quick and merge sort fall over to insertion
        # sort for small arrays.
        a = arange(101)
        b = a[::-1].copy()
        for kind in ['q', 'm', 'h'] :
            msg = "scalar sort, kind=%s" % kind
            c = a.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg
            c = b.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg

        # test complex sorts. These use the same code as the scalars
        # but the compare fuction differs.
        ai = a*1j + 1
        bi = b*1j + 1
        for kind in ['q', 'm', 'h'] :
            msg = "complex sort, real part == 1, kind=%s" % kind
            c = ai.copy();
            c.sort(kind=kind)
            assert (c == ai).all(), msg
            c = bi.copy();
            c.sort(kind=kind)
            assert (c == ai).all(), msg
        ai = a + 1j
        bi = b + 1j
        for kind in ['q', 'm', 'h'] :
            msg = "complex sort, imag part == 1, kind=%s" % kind
            c = ai.copy();
            c.sort(kind=kind)
            assert (c == ai).all(), msg
            c = bi.copy();
            c.sort(kind=kind)
            assert (c == ai).all(), msg

        # check axis handling. This should be the same for all type
        # specific sorts, so we only check it for one type and one kind
        a = array([[3, 2], [1, 0]])
        b = array([[1, 0], [3, 2]])
        c = array([[2, 3], [0, 1]])
        d = a.copy()
        d.sort(axis=0)
        assert (d == b).all(), "test sort with axis=0"
        d = a.copy()
        d.sort(axis=1)
        assert (d == c).all(), "test sort with axis=1"
        d = a.copy()
        d.sort()
        assert (d == c).all(), "test sort with default axis"

    def test_sort_corner_cases_string_records(self):
        skip('not implemented yet')
        from numpypy import array, dtype
        # test string sorts.
        s = 'aaaaaaaa'
        a = array([s + chr(i) for i in range(101)])
        b = a[::-1].copy()
        for kind in ['q', 'm', 'h'] :
            msg = "string sort, kind=%s" % kind
            c = a.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg
            c = b.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg


        # test record array sorts.
        dt =dtype([('f', float), ('i', int)])
        a = array([(i, i) for i in range(101)], dtype = dt)
        b = a[::-1]
        for kind in ['q', 'h', 'm'] :
            msg = "object sort, kind=%s" % kind
            c = a.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg
            c = b.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg

    def test_sort_unicode(self):
        from numpypy import array
        # test unicode sorts.
        s = 'aaaaaaaa'
        try:
            a = array([s + chr(i) for i in range(101)], dtype=unicode)
            b = a[::-1].copy()
        except:
            skip('unicode type not supported yet')
        for kind in ['q', 'm', 'h'] :
            msg = "unicode sort, kind=%s" % kind
            c = a.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg
            c = b.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg

    def test_sort_objects(self):
        # test object array sorts.
        from numpypy import empty
        try:
            a = empty((101,), dtype=object)
        except:
            skip('object type not supported yet')
        a[:] = list(range(101))
        b = a[::-1]
        for kind in ['q', 'h', 'm'] :
            msg = "object sort, kind=%s" % kind
            c = a.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg
            c = b.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg

    def test_sort_datetime(self):
        from numpypy import arange
        # test datetime64 sorts.
        try:
            a = arange(0, 101, dtype='datetime64[D]')
        except:
            skip('datetime type not supported yet')
        b = a[::-1]
        for kind in ['q', 'h', 'm'] :
            msg = "datetime64 sort, kind=%s" % kind
            c = a.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg
            c = b.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg

        # test timedelta64 sorts.
        a = arange(0, 101, dtype='timedelta64[D]')
        b = a[::-1]
        for kind in ['q', 'h', 'm'] :
            msg = "timedelta64 sort, kind=%s" % kind
            c = a.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg
            c = b.copy();
            c.sort(kind=kind)
            assert (c == a).all(), msg

    def test_sort_order(self):
        from numpypy import array, zeros
        from sys import byteorder
        # Test sorting an array with fields
        skip('not implemented yet')
        x1 = array([21, 32, 14])
        x2 = array(['my', 'first', 'name'])
        x3=array([3.1, 4.5, 6.2])
        r=zeros(3, dtype=[('id','i'),('word','S5'),('number','f')])
        r['id'] = x1
        r['word'] = x2
        r['number'] = x3

        r.sort(order=['id'])
        assert (r['id'] == [14, 21, 32]).all()
        assert (r['word'] == ['name', 'my', 'first']).all()
        assert max(abs(r['number'] - [6.2, 3.1, 4.5])) < 1e-6

        r.sort(order=['word'])
        assert (r['id'] == [32, 21, 14]).all()
        assert (r['word'] == ['first', 'my', 'name']).all()
        assert max(abs(r['number'] - [4.5, 3.1, 6.2])) < 1e-6

        r.sort(order=['number'])
        assert (r['id'] == [21, 32, 14]).all()
        assert (r['word'] == ['my', 'first', 'name']).all()
        assert max(abs(r['number'] - [3.1, 4.5, 6.2])) < 1e-6

        if byteorder == 'little':
            strtype = '>i2'
        else:
            strtype = '<i2'
        mydtype = [('name', 'S5'), ('col2', strtype)]
        r = array([('a', 1), ('b', 255), ('c', 3), ('d', 258)],
                     dtype= mydtype)
        r.sort(order='col2')
        assert (r['col2'] == [1, 3, 255, 258]).all()
        assert (r == array([('a', 1), ('c', 3), ('b', 255), ('d', 258)],
                                 dtype=mydtype)).all()

# tests from numpy/tests/test_regression.py
    def test_sort_bigendian(self):
        skip('not implemented yet')
        from numpypy import array, dtype
        a = array(range(11),dtype='float64')
        c = a.astype(dtype('<f8'))
        c.sort()
        assert max(abs(a-c)) < 1e-32

    def test_string_sort_with_zeros(self):
        skip('not implemented yet')
        from numpypy import fromstring
        """Check sort for strings containing zeros."""
        x = fromstring("\x00\x02\x00\x01", dtype="S2")
        y = fromstring("\x00\x01\x00\x02", dtype="S2")
        x.sort(kind='q')
        assert (x == y).all()

    def test_string_mergesort(self):
        import numpypy as np
        import sys
        x = np.array(['a'] * 32)
        if '__pypy__' in sys.builtin_module_names:
            exc = raises(NotImplementedError, "x.argsort(kind='m')")
            assert 'non-numeric types' in exc.value.message
        else:
            assert (x.argsort(kind='m') == np.arange(32)).all()
