///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "AbstractFileColumnWriter.h"

namespace AtomViz {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(AbstractFileColumnWriter, AtomsFileWriter)

/******************************************************************************
 * Sets the mapping between data columns in the output file and
 * the DataChannel objects in the source AtomsObject.
 *****************************************************************************/
void AbstractFileColumnWriter::setChannelMapping(const ChannelColumnMapping& mapping)
{
	_channelMapping = mapping;
	
	// Remember the mapping for the next time.
	QSettings settings;
	settings.beginGroup("atomviz/io/channelmapping/");
	settings.setValue(pluginClassDescriptor()->name(), mapping.toByteArray());
	settings.endGroup();
}

/******************************************************************************
 * Prepares the writer based on the scene to be exported.
 *****************************************************************************/
bool AbstractFileColumnWriter::prepareScene(DataSet* dataset, bool suppressDialogs)
{
	// Use the last mapping by default.
	QSettings settings;
	settings.beginGroup("atomviz/io/channelmapping/");
	if(settings.contains(pluginClassDescriptor()->name())) {
		try {
			_channelMapping.fromByteArray(settings.value(pluginClassDescriptor()->name()).toByteArray());
		}
		catch(Exception& ex) {
			ex.prependGeneralMessage(tr("Failed to load last used channel-to-column mapping from application settings store."));
			ex.logError();
		}
	}
	settings.endGroup();
	
	return AtomsFileWriter::prepareScene(dataset, suppressDialogs);
}

/******************************************************************************
 * Saves the class' contents to the given stream.
 *****************************************************************************/
void AbstractFileColumnWriter::saveToStream(ObjectSaveStream& stream)
{
	AtomsFileWriter::saveToStream(stream);	
	stream.beginChunk(0x442AAB);
	// Save the writer's data channel to column mapping into the scene file.
	_channelMapping.saveToStream(stream);
	stream.endChunk(); 
}

/******************************************************************************
 * Loads the class' contents from the given stream.
 *****************************************************************************/
void AbstractFileColumnWriter::loadFromStream(ObjectLoadStream& stream)
{
	AtomsFileWriter::loadFromStream(stream);
	stream.expectChunk(0x442AAB);
	// Load the writer's data channel to column mapping from the scene file.
	_channelMapping.loadFromStream(stream);	
	stream.closeChunk();
}

/******************************************************************************
 * Creates a copy of this object.
 *****************************************************************************/
RefTarget::SmartPtr AbstractFileColumnWriter::clone(bool deepCopy, CloneHelper& cloneHelper)
{
	// Let the base class create an instance of this class.
	AbstractFileColumnWriter::SmartPtr clone = static_object_cast<AbstractFileColumnWriter>(AtomsFileWriter::clone(deepCopy, cloneHelper));	
	
	clone->_channelMapping = this->_channelMapping;
		
	return clone;
}

};	// End of namespace AtomViz
