/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file alice.c
 * \brief This file contains Turbolink router related functions
 */

#include <ffgtk.h>
#include <router/common.h>

/** logged in counter */
static gchar nLoggedIn = 0;

/**
 * \brief Detect alicebox
 * \param psProfile profile structure
 * \return error code
 */
static int aliceBoxDetect( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gint nError;
	gchar anUrl[ BUF_SIZE ];

	snprintf( anUrl, sizeof( anUrl ), "%s/web.cgi", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEJAR, "/tmp/cookies.txt" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEFILE, "/tmp/cookies.txt" );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_INFO, "No alicebox!\n" );
		freeHandler( psHandler );
		return nError;
	}

	//saveData( "./alice-detect.html", psHandler -> pnData );

	if ( g_strcasestr( psHandler -> pnData, "alicebox" ) ) {
		nError = 0;
		Debug( KERN_DEBUG, "alicebox found!\n" );
	} else {
		nError = 1;
		Debug( KERN_DEBUG, "No alicebox\n" );
	}
	freeHandler( psHandler );

	return nError;
}

/**
 * \brief Login to alicebox
 * \param psProfile profile structure
 * \return 0 on success, negative values (seconds to sleep) on error
 */
static int aliceBoxLogin( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler = NULL;
	gint nError = -1;
	gchar anUrl[ BUF_SIZE ];
	const gchar *pnPassword = routerGetPassword( psProfile );

	if ( nLoggedIn != 0 ) {
		nLoggedIn++;
		//Debug( KERN_DEBUG, "LoggedIn: %d\n", nLoggedIn );
		return 0;
	}

	if ( pnPassword == NULL ) {
		goto end;
	}

	/* login */
	snprintf( anUrl, sizeof( anUrl ), "%s/web.cgi", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );

	setPostData( psHandler, "controller=Overview&action=Login&id=0&idTextPassword=%s", pnPassword );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEJAR, "/tmp/cookies.txt" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEFILE, "/tmp/cookies.txt" );

	nError = readUrl( psHandler, psProfile );
	//saveData( "./alice-login.html", psHandler -> pnData );

	if ( findString( psHandler -> pnData, 0, "cLogout" ) == -1 ) {
		Debug( KERN_WARNING, "Login failed, wrong password!\n" );
	} else {
		nError = 0;
		//Debug( KERN_DEBUG, "Login successful!\n" );
		nLoggedIn++;
	}
	freeHandler( psHandler );

	//Debug( KERN_DEBUG, "LoggedIn: %d\n", nLoggedIn );

end:
	/* If we failed to login due to a wrong password, lock router structure */
	if ( nError < 0 ) {
		routerLock( psProfile, TRUE );
	}

	return nError;
}

/**
 * \brief Logout from alicebox
 * \param psProfile profile structure
 * \return error code
 */
static int aliceBoxLogout( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ BUF_SIZE ];
	gint nError = -1;

	nLoggedIn--;

	if ( nLoggedIn > 0 ) {
		return 0;
	}
	if ( nLoggedIn < 0 ) {
		nLoggedIn = 0;
	}

	/* Logout */
	snprintf( anUrl, sizeof( anUrl ), "%s/web.cgi?controller=Overview&action=Logout", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );

	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEJAR, "/tmp/cookies.txt" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEFILE, "/tmp/cookies.txt" );

	nError = readUrl( psHandler, psProfile );

	//saveData( "./alice-logout.html", psHandler -> pnData );
	freeHandler( psHandler );

	return nError;
}

/**
 * \brief Detect alicebox firmware
 * \param psProfile profile structure
 * \return error code
 */
static int aliceBoxDetectFirmware( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler;
	gchar anUrl[ 1024 ];
	gint nError = 1;

	Debug( KERN_DEBUG, "Detecting alicebox firmware...\n" );

	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	psProfile -> sRouterInfo.pnFirmware = NULL;

	/* status page */
	snprintf( anUrl, sizeof( anUrl ), "%s/web.cgi?controller=Overview&action=IndexOverview&id=0", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );

	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEJAR, "/tmp/cookies.txt" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEFILE, "/tmp/cookies.txt" );

	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_DEBUG, "No data\n" );
		freeHandler( psHandler );
		goto end;
	}

	//saveData( "./alice-firmware.html", psHandler -> pnData );

	GRegex *psReg;
	GMatchInfo *psInfo;
	gchar *pnRegex = "Alice (\\s|\\w)*\\s\\d+";

	/* Search for device type */
	psReg = g_regex_new( pnRegex, 0, 0, NULL );

	g_regex_match( psReg, psHandler -> pnData, 0, &psInfo );

	if ( g_match_info_matches( psInfo ) ) {
		gchar *pnWord = g_match_info_fetch( psInfo, 0 );
		Debug( KERN_DEBUG, "pnWord: %s\n", pnWord );

		if ( pnWord[ strlen( pnWord ) - 1 ] == '\n' ) {
			pnWord[ strlen( pnWord ) - 1 ] = '\0';
		}
		psProfile -> sRouterInfo.pnFirmware = convertEntities( pnWord );
		Debug( KERN_DEBUG, "Type: [%s]\n", psProfile -> sRouterInfo.pnFirmware );
		if ( !strncmp( psProfile -> sRouterInfo.pnFirmware, "Alice IAD WLAN 3231", 19 ) ) {
			psProfile -> sRouterInfo.nType = 1;
		}
		g_free( pnWord );
	}
	g_match_info_free( psInfo );

	g_regex_unref( psReg );

	/* Search for firmware */
	pnRegex = "\\d.\\d\\d-\\d.\\d\\d.\\d\\w";
	psReg = g_regex_new( pnRegex, 0, 0, NULL );

	g_regex_match( psReg, psHandler -> pnData, 0, &psInfo );

	if ( g_match_info_matches( psInfo ) ) {
		gchar *pnWord = g_match_info_fetch( psInfo, 0 );
		Debug( KERN_DEBUG, "pnWord: %s\n", pnWord );

		if ( pnWord[ strlen( pnWord ) - 1 ] == '\n' ) {
			pnWord[ strlen( pnWord ) - 1 ] = '\0';
		}
		psProfile -> sRouterInfo.pnFirmware = convertEntities( pnWord );
		g_free( pnWord );
	}
	g_match_info_free( psInfo );

	g_regex_unref( psReg );

	routerSetFirmware( psProfile, psProfile -> sRouterInfo.pnFirmware );

end:
	//commonParseFirmware( psProfile );

	if ( psProfile -> sRouterInfo.pnFirmware == NULL || strlen( psProfile -> sRouterInfo.pnFirmware ) == 0 ) {
		Debug( KERN_WARNING, "Could not parse alicebox firmware. set type to 0 and enabled all ports as fallback\n" );
		nError = 1;
	} else {
		nError = 0;
	}

	freeHandler( psHandler );
	routerLogout( psProfile );

	return nError;
}

/**
 * \brief Disconnect and quick reconnect
 * \param psProfile profile structure
 * \return error code
 */
static int aliceBoxReconnect( struct sProfile *psProfile ) {
	struct sUrlHandler *psHandler = NULL;
	gint nError = -1;
	gchar anUrl[ BUF_SIZE ];

	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return nError;
	}

	/* disconnect */
	snprintf( anUrl, sizeof( anUrl ), "%s/web.cgi", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );

	setPostData( psHandler, "controller=Overview&action=StopPppoeMan&id=0" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEJAR, "/tmp/cookies.txt" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEFILE, "/tmp/cookies.txt" );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_WARNING, "Could not stop connection!\n" );
	}

	/* connect */
	setPostData( psHandler, "controller=Overview&action=StartPppoeMan&id=0" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEJAR, "/tmp/cookies.txt" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEFILE, "/tmp/cookies.txt" );
	nError = readUrl( psHandler, psProfile );
	if ( nError != 0 ) {
		Debug( KERN_WARNING, "Could not start connection!\n" );
	}

	freeHandler( psHandler );

	routerLogout( psProfile );


	return nError;
}

/**
 * \brief Get external ip
 * \param psProfile profile structure
 * \return external ip address or NULL on error
 */
static gchar *aliceBoxGetIp( struct sProfile *psProfile ) {
	gchar *pnIp = NULL;
	gint nStart, nEnd;
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ 256 ];
	gint nError = -1;

	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return NULL;
	}
	nError = -1;

	snprintf( anUrl, sizeof( anUrl ), "%s/web.cgi?controller=Internet&action=IndexInfoConnection", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEJAR, "/tmp/cookies.txt" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEFILE, "/tmp/cookies.txt" );
	nError = readUrl( psHandler, psProfile );

	if ( nError == 0 ) {
		//saveData( "./alice_getip.html", psHandler -> pnData );
		/* Get external ip */
		nStart = findString( psHandler -> pnData, 0, "IP-Adresse:" );
		if ( nStart == -1 ) {
			goto end;
		}

		nStart += 31;
		nEnd = findString( psHandler -> pnData, nStart, "</td>" );
		if ( nEnd == -1 ) {
			goto end;
		}

		//Debug( KERN_DEBUG, "nStart %d, nEnd %d\n", nStart, nEnd );

		pnIp = getSubString( psHandler -> pnData, nStart, nEnd - nStart - 1 );
		//Debug( KERN_DEBUG, "IP: '%s'\n", pnIp );
	}

end:
	freeHandler( psHandler );

	routerLogout( psProfile );

	return pnIp;
}

/**
 * \brief Get maximal speed
 * \param psProfile profile structure
 * \param pnMaxUp maximal up speed
 * \param pnMaxDown maximual down speed
 */
static void aliceBoxGetSpeed( struct sProfile *psProfile, gint *pnMaxUp, gint *pnMaxDown ) {
	gint nStart, nEnd;
	struct sUrlHandler *psHandler = NULL;
	gchar anUrl[ 256 ];
	gint nError = -1;
	gchar *pnTmp;

	nError = routerLogin( psProfile );
	if ( nError != 0 ) {
		return;
	}

	snprintf( anUrl, sizeof( anUrl ), "%s/web.cgi?controller=Internet&action=IndexInfoConnection", routerGetHost( psProfile ) );
	psHandler = urlHandler( anUrl, 80 );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEJAR, "/tmp/cookies.txt" );
	curl_easy_setopt( psHandler -> psHandle, CURLOPT_COOKIEFILE, "/tmp/cookies.txt" );
	nError = readUrl( psHandler, psProfile );

	if ( nError == 0 ) {
		nStart = findString( psHandler -> pnData, 0, "Netto-Datenrate:" );

		nEnd = 0;
		if ( nStart != -1 ) {
			nStart += 35;
			nEnd = findString( psHandler -> pnData, nStart, " kbit/s" );
			if ( nEnd != -1 ) {
				pnTmp = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
				*pnMaxDown = atoi( pnTmp ) * 1024;
				g_free( pnTmp );
			} else {
				nEnd = 0;
			}
		}

		nStart = findString( psHandler -> pnData, nEnd, "<td>" );

		if ( nStart != -1 ) {
			nStart += 4;
			nEnd = findString( psHandler -> pnData, nStart, " kbit/s" );
			if ( nEnd != -1 ) {
				pnTmp = getSubString( psHandler -> pnData, nStart, nEnd - nStart );
				*pnMaxUp = atoi( pnTmp ) * 1024;
				g_free( pnTmp );
			}
		}
	}

	freeHandler( psHandler );

	routerLogout( psProfile );
}

/**
 * \brief Get phone settings (country/city prefix) of alicebox (UNSUPPORTED)
 * \param psProfile profile structure
 * \return error code, -1
 */
static int aliceBoxGetPhoneSettings( struct sProfile *psProfile ) {
	return 0;
}

/**
 * \brief Get supported ports of alicebox (UNSUPPORTED)
 * \param psProfile profile structure
 * \return error code, -1
 */
static int aliceBoxGetActivePorts( struct sProfile *psProfile ) {
	return 0;
}

/**
 * \brief Get alicebox name
 * \param psProfile profile structure
 * \return alicebox name or NULL for unknown
 */
static const char *aliceBoxGetName( struct sProfile *psProfile ) {
	switch ( psProfile -> sRouterInfo.nType ) {
		case 1:
			return "Alice IAD WLAN 3231";
		default:
			break;
	}

	return NULL;
}

/**
 * \brief Get alicebox version
 * \param psProfile profile structure
 * \return alicebox version
 */
static const gchar *aliceBoxGetVersion( struct sProfile *psProfile ) {
	if ( psProfile -> sRouterInfo.pnFirmware != NULL ) {
		return psProfile -> sRouterInfo.pnFirmware;
	}

	return "Unknown";
}

/**
 * \brief Init alice box info structure
 * \param psProfile profile structure
 * \return error code 0
 */
static int aliceBoxInit( struct sProfile *psProfile ) {
	memset( &psProfile -> sRouterInfo, 0, sizeof( psProfile -> sRouterInfo ) );
	psProfile -> sRouterInfo.pnFirmware = ( gchar * ) routerGetFirmware( psProfile );
	psProfile -> sRouterInfo.nType = 1;
	//commonParseFirmware( psProfile );

	return 0;
}

/** AliceBox router structure */
struct sRouter sAliceBox = {
	"AliceBox",
	aliceBoxDetect,
	aliceBoxInit,
	aliceBoxLogin,
	aliceBoxLogout,
	aliceBoxDetectFirmware,
	aliceBoxGetPhoneSettings,
	aliceBoxGetActivePorts,
	NULL,
	NULL,
	NULL,
	NULL,
	aliceBoxReconnect,
	aliceBoxGetIp,
	aliceBoxGetSpeed,
	NULL,
	NULL,
	aliceBoxGetName,
	aliceBoxGetVersion,
	NULL,
	NULL,
};
